/*! \file 
**********************************************************************************
*Title:                         Discretix OMA DRM v2 Secure RO Handling ATP Test source file
*
* Filename:                     ODRM_SecROHandling_RoPreProcessing_ATP.c 
*
* 
* Created:                      17.03.2009
*
*
* \Author                      Shelly Lerman
*
* \Remarks
*           Copyright (C) 2007 by Discretix Technologies Ltd. All Rights reserved.
**********************************************************************************/
/************* Include Files ***********************************/
#include "DX_VOS_BaseTypes.h"
#include "CRYS.h"
#include "tlk_odrm_types.h"
#include "KMNG_Defs.h" 
#include "ODRM_TLK_ATP_UTIL.h"
#include "KMNG_API.h"
#include "ODRM_SecROHandling_RoPreProcessing_ATP.h"
#include "tlk_odrm_api.h"
#include "DX_VOS_Mem.h"
#include "ODRM_SecROHandling_data.h"
#include "DX_VOS_Stdio.h"
#include "CRYS_KMNG.h"     
#include "tlk_odrm_errors.h"
#include "tlk_cert_types.h"
#include "tlk_sclk_api_types.h"
#include "DX_VOS_String.h"
#include "KMNG_API.h"
#include "tlk_sclk_api.h"
#include "MW_ATP_UTIL_funcs.h"
#include <stdio.h>
#include "ODRM_SecROHandling_Utils_ATP.h"


/*****************************************************************************
* Function Name:                                                           
*  TST_ATP_SEC_RO_PreProcessingAndActivation
* 
* Inputs:
*  None
*
* Outputs:
*  DxError_t - Function error return                                        
*
* TEST DESCRIPTION:
*  To test License Pre-Processing and Activation using device private key in Secure RO Handling 
* Algorithm:
* 1. Get workspace size. 
* 2. Create key ring and initialize it with device private key
* 3. Initialize Secure Clock descriptor
* 4. Call to  TLK_ODRM_VerifyDeviceRoBeforeInstall() API with the device private key . 
* 5. Activate content license using  TLK_ODRM_RetrieveKey() with KMAC and KREK keys from step 4 and secure clock descriptor from step 3.
* 6. Validate that  this is correct RO key.
*    6.1 Encrypt Plain text with the activated RO key from TLK_ODRM_RetrieveKey().
*    6.2 compare it to the Encrypted plain text with the 
*        original RO key
*******************************************************************************/   
DxError_t TST_ATP_SEC_RO_PreProcessingAndActivation(void)
{

    DxUint32_t TST_Error = 0;
    DxUint8_t index = 0; 
    TLK_ODRM_KMNGKeyRing_t keyRingDeviceStruct = {0};

    TLK_ODRM_KMNGKeyRing_t keyRingKmacKrecStruct = {0};

    TLK_ODRM_KMNGKey_t  TST_KeyDevice = {0};
    TLK_ODRM_KMNGKey_t  TST_kmacKey = {0};
    TLK_ODRM_KMNGKey_t  TST_krecKey = {0};
    TLK_ODRM_Buffer_t   TST_RightsXML = {0};
    DxUint32_t  TST_KeyRingSize = 0;

    KMNG_AES_WrappedKey_t  TST_WrappedKey;

    KMNG_UserSpecificKeyData_t  TST_UserSpecificKeyData = {0,0};
    TLK_SCLK_ServiceClockDescriptor_t    TST_ServiceDescr = {0};

    DxUint32_t TST_NumOfSymKeys = 0;
    DxUint32_t TST_NumOfRSAKeys = 0;
    DxUint32_t TST_NumOfDHKeys  = 0;
  
    /*******************Set password to RI ID ****************/
    TLK_ODRM_Buffer_t passwordBuff;
    passwordBuff.buff_ptr        = (DxUint8_t*)TST_RiIDKey;
    passwordBuff.buffSizeInBytes = DX_VOS_StrLen(TST_RiIDKey);

    /*Initialize rights  struct*/
    TST_RightsXML.buffSizeInBytes = DX_VOS_StrLen(RightsXmlStr);
    TST_RightsXML.buff_ptr = (DxByte*)RightsXmlStr;

    TST_KRING_Clean();

    ATP_LOG_TST_PRINT((MW_ATP_MSG,"\n***************************************************** \n"));
    ATP_LOG_TST_PRINT((MW_ATP_MSG,"***** TST_ATP_SEC_RO_PreProcessingAndActivation *****\n"));     
    ATP_LOG_TST_PRINT((MW_ATP_MSG,"*****************************************************\n \n"));

    /* 1. Get work space size */
    TST_Error = TLK_ODRM_WorkspaceSizeGet(TLK_ODRM_MIN_WORKSPACE, &RO_TST_gWorkspaceSizeInBytes);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_ODRM_WorkspaceSizeGet",
                               "TST_ATP_SEC_RO_PreProcessingAndActivation", 
                               "TST_ATP_SEC_RO_PreProcessingAndActivation");


    if(RO_TST_gWorkspaceSizeInBytes < ODRMTLK_TST_WORKSPACE_SIZE)
       TST_Error = TST_FAIL;

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "Insufficient Workspace Size",
                               "TST_ATP_SEC_RO_PreProcessingAndActivation",
                               "TST_ATP_SEC_RO_PreProcessingAndActivation");


    /* 2. Create key ring and initialize it with device private key */
    TST_NumOfSymKeys = 0;
    TST_NumOfRSAKeys = 1;
    TST_NumOfDHKeys  = 0;
    TST_Error =  KMNG_KeyRingInit(passwordBuff.buffSizeInBytes,
                                  passwordBuff.buff_ptr,
                                  TST_NumOfSymKeys,
                                  TST_NumOfRSAKeys,
                                  TST_NumOfDHKeys,
                                  TST_gKeyRingDevice);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "KMNG_KeyRingInit - TST_gKeyRingDevice",
                               "TST_ATP_SEC_RO_PreProcessingAndActivation",
                               "TST_ATP_SEC_RO_PreProcessingAndActivation");

    TST_NumOfSymKeys = 2;
    TST_NumOfRSAKeys = 0;
    TST_NumOfDHKeys  = 0;
    TST_Error = KMNG_KeyRingInit(passwordBuff.buffSizeInBytes,
                                  passwordBuff.buff_ptr,
                                  TST_NumOfSymKeys,
                                  TST_NumOfRSAKeys,
                                  TST_NumOfDHKeys,
                                  TST_gKeyRingKmacKrec);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "KMNG_KeyRingInit - TST_gKeyRingKmacKrec",
                               "TST_ATP_SEC_RO_PreProcessingAndActivation",
                               "TST_ATP_SEC_RO_PreProcessingAndActivation");

    TST_KeyDevice.keyPassword_ptr =passwordBuff.buff_ptr;
    TST_KeyDevice.keyPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    TST_kmacKey.keyPassword_ptr =passwordBuff.buff_ptr;
    TST_kmacKey.keyPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    TST_krecKey.keyPassword_ptr =passwordBuff.buff_ptr;
    TST_krecKey.keyPasswordLenInBytes = passwordBuff.buffSizeInBytes;


    TST_Error = KMNG_ImportRsaUserKey(passwordBuff.buff_ptr, 
                                      passwordBuff.buffSizeInBytes,
                                      passwordBuff.buff_ptr, 
                                      passwordBuff.buffSizeInBytes,
                                      KMNG_KeyTypeRSAPair,                    /*keyType*/
                                      ODRMTLK_TST_RSA_1024_KEY_SIZE_IN_BYTES, /*keySize*/ 
                                      KMNG_KEY_USAGE_ODRM | KMNG_KEY_USAGE_SIGNING,                    /*keyUsage*/
                                      TLK_ODRM_KEY_RESTRICTION,               /*keyRestriction*/
                                      TST_UserSpecificKeyData,                    /*UserSpecificKeyData*/
                                      PrivModulus,                           /*modulus_ptr*/
                                      MODULUS_SIZE,                       /*modulusSize*/
                                      PubExponent,                       /*E_ptr*/
                                      PUB_EXP_SIZE,                    /*E_Size*/
                                      PrivExponent,                       /*D_ptr*/
                                      PRIV_EXP_SIZE,                   /*D_Size*/
                                      &TST_KeyDevice.keyId,
                                      TST_gKeyRingDevice);


    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "KMNG_ImportRsaUserKey ",
                               "TST_ATP_SEC_RO_PreProcessingAndActivation",
                               "TST_ATP_SEC_RO_PreProcessingAndActivation");
    
    keyRingDeviceStruct.keyKmngRing_ptr = TST_gKeyRingDevice;
    keyRingDeviceStruct.keyKmngRingPassword_ptr = passwordBuff.buff_ptr;
    keyRingDeviceStruct.keyKmngRingPasswordLenInBytes = passwordBuff.buffSizeInBytes;
    
    keyRingKmacKrecStruct.keyKmngRing_ptr = TST_gKeyRingKmacKrec;
    keyRingKmacKrecStruct.keyKmngRingPassword_ptr = passwordBuff.buff_ptr;
    keyRingKmacKrecStruct.keyKmngRingPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    /*3.Or initialize secure clock descriptor*/
    TST_Error = TLK_SCLK_Init(RO_TST_gWorkspace, RO_TST_gWorkspaceSizeInBytes);
    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_SCLK_Init",
                               "TST_ATP_SEC_RO_PreProcessingAndActivation",
                               "TST_ATP_SEC_RO_PreProcessingAndActivation");

   TST_Error = TST_SCLK_Init(&TST_ServiceDescr);
   TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TST_SCLK_Init",
                              "TST_ATP_SEC_RO_PreProcessingAndActivation",
                              "TST_ATP_SEC_RO_PreProcessingAndActivation");

 
   
    /*4. Call to  TLK_ODRM_VerifyDeviceRoBeforeInstall() API with the device private key */
    TST_Error = TLK_ODRM_VerifyDeviceRoBeforeInstall(&TST_RightsXML,
                                                     &keyRingDeviceStruct,
                                                     &TST_KeyDevice,
                                                     &keyRingKmacKrecStruct,
                                                     &TST_kmacKey,
                                                     &TST_krecKey,
                                                     RO_TST_gWorkspace,
                                                     RO_TST_gWorkspaceSizeInBytes);
 

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_ODRM_VerifyDeviceRoBeforeInstall",
                               "TST_ATP_SEC_RO_PreProcessingAndActivation",
                               "TST_ATP_SEC_RO_PreProcessingAndActivation");


    /*5. Activate content license using  TLK_ODRM_RetrieveKey() with KMAC 
         and KREK keys from step 4 and secure clock descriptor from step 3.*/
    TST_Error = TLK_ODRM_RetrieveKey(&TST_RightsXML,
                                     &keyRingKmacKrecStruct,
                                     &TST_kmacKey,
                                     &TST_krecKey,
                                     "cid:cid6093_jpg",
                                     &TST_ServiceDescr,
                                     DX_NULL,
                                     DX_NULL,
                                     DX_NULL,
                                     DX_NULL,
                                     0,
                                     TST_WrappedKey,
                                     RO_TST_gWorkspace,
                                     RO_TST_gWorkspaceSizeInBytes);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_ODRM_RetrieveKey",
                               "TST_ATP_SEC_RO_PreProcessingAndActivation",
                               "TST_ATP_SEC_RO_PreProcessingAndActivation");

    /* 6.Validate that  this is correct RO key.*/

    /*6.1  Encrypt Plain text with the activated RO key from    TLK_ODRM_RetrieveKey().*/     
    TST_Error = CRYS_KMNG_AES(TST_WrappedKey,
                              IvCounter,
                              CRYS_AES_Encrypt,
                              CRYS_AES_ECB_mode,
                              TST_PlainBuffCid6093,
                              ODRMTLK_TST_AES_KEY_SIZE,
                              TST_DataoutBuff);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "CRYS_KMNG_AES",
                               "TST_ATP_SEC_RO_PreProcessingAndActivation",
                               "TST_ATP_SEC_RO_PreProcessingAndActivation");

    /* 6.2 compare it to the Encrypted plain text with the original RO key.*/
    TST_Error = DX_VOS_MemCmp(TST_DataoutBuff, TST_EncryptBuffCid6093, TEMP_BUFF_LEN);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "DX_VOS_MemCmp",
                               "TST_ATP_SEC_RO_PreProcessingAndActivation",
                               "TST_ATP_SEC_RO_PreProcessingAndActivation");

EXIT_ON_ERROR:
    TLK_SCLK_Terminate(RO_TST_gWorkspace, RO_TST_gWorkspaceSizeInBytes);
    return TST_Error;
}


/*****************************************************************************
* Function Name:                                                           
*  TST_ATP_SEC_RO_PreProcessingWithInvalidFlag
* 
* Inputs:
*  None
*
* Outputs:
*  DxError_t - Function error return                                        
*
* TEST DESCRIPTION:
*  To test license pre-processing with invalid usage flag of device private key
* Algorithm:
*1. Get workspace size. 
*2. Create key ring and initialize it 
*3. Import a device private key with invalid usage flag
*4. Call to TLK_ODRM_VerifyDeviceRoBeforeInstall()  API with invalid usage flag of device private key
*5. Verify that TLK_ODRM_RC_ERROR_INCORRECT_KEY_PERMISSIONS error was returned
*******************************************************************************/   
DxError_t TST_ATP_SEC_RO_PreProcessingWithInvalidFlag(void)
{
    DxUint32_t TST_Error = 0;
    DxUint8_t index = 0; 
    TLK_ODRM_KMNGKeyRing_t keyRingDeviceStruct = {0};

    TLK_ODRM_KMNGKeyRing_t keyRingKmacKrecStruct = {0};

    TLK_ODRM_KMNGKey_t  TST_KeyDevice = {0};
    TLK_ODRM_KMNGKey_t  TST_kmacKey = {0};
    TLK_ODRM_KMNGKey_t  TST_krecKey = {0};
    TLK_ODRM_Buffer_t   TST_RightsXML = {0};
    DxUint32_t  TST_KeyRingSize = 0;


    KMNG_UserSpecificKeyData_t  TST_UserSpecificKeyData = {0,0};

    DxUint32_t TST_NumOfSymKeys = 0;
    DxUint32_t TST_NumOfRSAKeys = 0;
    DxUint32_t TST_NumOfDHKeys  = 0;

    /*******************Set password to RI ID****************/
    TLK_ODRM_Buffer_t passwordBuff;
    passwordBuff.buff_ptr        = TST_RiIDKey;
    passwordBuff.buffSizeInBytes = DX_VOS_StrLen(TST_RiIDKey);
    TST_KRING_Clean();

    ATP_LOG_TST_PRINT((MW_ATP_MSG,"\n******************************************************* \n"));
    ATP_LOG_TST_PRINT((MW_ATP_MSG,"***** TST_ATP_SEC_RO_PreProcessingWithInvalidFlag *****\n"));     
    ATP_LOG_TST_PRINT((MW_ATP_MSG,"*******************************************************\n \n"));


    /*Initialize rights  struct*/
    TST_RightsXML.buffSizeInBytes = DX_VOS_StrLen(RightsXmlStr);
    TST_RightsXML.buff_ptr = (DxByte*)RightsXmlStr;

    /* 1. Get work space size */
    TST_Error = TLK_ODRM_WorkspaceSizeGet(TLK_ODRM_MIN_WORKSPACE, &RO_TST_gWorkspaceSizeInBytes);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_ODRM_WorkspaceSizeGet",
                               "TST_ATP_SEC_RO_PreProcessingWithInvalidFlag", 
                               "TST_ATP_SEC_RO_PreProcessingWithInvalidFlag");


    if(RO_TST_gWorkspaceSizeInBytes < ODRMTLK_TST_WORKSPACE_SIZE)
        TST_Error = TST_FAIL;

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "Insufficient Workspace Size",
                               "TST_ATP_SEC_RO_PreProcessingWithInvalidFlag",
                               "TST_ATP_SEC_RO_PreProcessingWithInvalidFlag");


    /* 2. Create key ring and initialize it with device private key */
    TST_NumOfSymKeys = 0;
    TST_NumOfRSAKeys = 1;
    TST_NumOfDHKeys  = 0;
    TST_Error =  KMNG_KeyRingInit(passwordBuff.buffSizeInBytes,
                                  passwordBuff.buff_ptr,
                                  TST_NumOfSymKeys,
                                  TST_NumOfRSAKeys,
                                  TST_NumOfDHKeys,
                                  TST_gKeyRingDevice);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "KMNG_KeyRingInit - TST_gKeyRingDevice",
                               "TST_ATP_SEC_RO_PreProcessingWithInvalidFlag",
                               "TST_ATP_SEC_RO_PreProcessingWithInvalidFlag");

    TST_NumOfSymKeys = 2;
    TST_NumOfRSAKeys = 0;
    TST_NumOfDHKeys  = 0;
    TST_Error = KMNG_KeyRingInit(passwordBuff.buffSizeInBytes,
                                 passwordBuff.buff_ptr,
                                 TST_NumOfSymKeys,
                                 TST_NumOfRSAKeys,
                                 TST_NumOfDHKeys,
                                 TST_gKeyRingKmacKrec);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "KMNG_KeyRingInit - TST_gKeyRingKmacKrec",
                               "TST_ATP_SEC_RO_PreProcessingWithInvalidFlag",
                               "TST_ATP_SEC_RO_PreProcessingWithInvalidFlag");

    TST_KeyDevice.keyPassword_ptr =passwordBuff.buff_ptr;
    TST_KeyDevice.keyPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    TST_kmacKey.keyPassword_ptr =passwordBuff.buff_ptr;
    TST_kmacKey.keyPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    TST_krecKey.keyPassword_ptr =passwordBuff.buff_ptr;
    TST_krecKey.keyPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    /*3. Import a device private key with invalid usage flag*/
    TST_Error = KMNG_ImportRsaUserKey(passwordBuff.buff_ptr, 
                                      passwordBuff.buffSizeInBytes,
                                      passwordBuff.buff_ptr, 
                                      passwordBuff.buffSizeInBytes,
                                      KMNG_KeyTypeRSAPair,                    /*keyType*/
                                      ODRMTLK_TST_RSA_1024_KEY_SIZE_IN_BYTES, /*keySize*/ 
                                      KMNG_KEY_USAGE_LEGACY,                    /*keyUsage*/
                                      TLK_ODRM_KEY_RESTRICTION,               /*keyRestriction*/
                                      TST_UserSpecificKeyData,                    /*UserSpecificKeyData*/
                                      PrivModulus,                           /*modulus_ptr*/
                                      MODULUS_SIZE,                       /*modulusSize*/
                                      PubExponent,                       /*E_ptr*/
                                      PUB_EXP_SIZE,                    /*E_Size*/
                                      PrivExponent,                       /*D_ptr*/
                                      PRIV_EXP_SIZE,                   /*D_Size*/
                                      &TST_KeyDevice.keyId,
                                      TST_gKeyRingDevice);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "KMNG_ImportRsaUserKey ",
                               "TST_ATP_SEC_RO_PreProcessingWithInvalidFlag",
                               "TST_ATP_SEC_RO_PreProcessingWithInvalidFlag");

    keyRingDeviceStruct.keyKmngRing_ptr = TST_gKeyRingDevice;
    keyRingDeviceStruct.keyKmngRingPassword_ptr = passwordBuff.buff_ptr;
    keyRingDeviceStruct.keyKmngRingPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    keyRingKmacKrecStruct.keyKmngRing_ptr = TST_gKeyRingKmacKrec;
    keyRingKmacKrecStruct.keyKmngRingPassword_ptr = passwordBuff.buff_ptr;
    keyRingKmacKrecStruct.keyKmngRingPasswordLenInBytes = passwordBuff.buffSizeInBytes;


    /*4. Call to  TLK_ODRM_VerifyDeviceRoBeforeInstall() API with the device private key . */
    TST_Error = TLK_ODRM_VerifyDeviceRoBeforeInstall(&TST_RightsXML,
                                                     &keyRingDeviceStruct,
                                                     &TST_KeyDevice,
                                                     &keyRingKmacKrecStruct,
                                                     &TST_kmacKey,
                                                     &TST_krecKey,
                                                     RO_TST_gWorkspace,
                                                     RO_TST_gWorkspaceSizeInBytes);

    /*5. Verify that TLK_ODRM_RC_ERROR_CRYS_KMNG error was returned */
    if(TST_Error != TLK_ODRM_RC_ERROR_INCORRECT_KEY_PERMISSIONS)
        TST_Error = TST_FAIL;
    else
    TST_Error = DX_SUCCESS;
    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_ODRM_VerifyDeviceRoBeforeInstall with invalid usage flag. Test ",
                               "TST_ATP_SEC_RO_PreProcessingWithInvalidFlag",
                               "TST_ATP_SEC_RO_PreProcessingWithInvalidFlag");

EXIT_ON_ERROR:
    TLK_SCLK_Terminate(RO_TST_gWorkspace, RO_TST_gWorkspaceSizeInBytes);
    return TST_Error;
}


/*****************************************************************************
* Function Name:                                                           
*  TST_ATP_SEC_RO_PreProcessingWithInvalidRO
* 
* Inputs:
*  None
*
* Outputs:
*  DxError_t - Function error return                                        
*
* TEST DESCRIPTION:
*  To test license pre-processing with invalid RO XML
* Algorithm:
*1. Get workspace size. 
*2. Create key ring and initialize it 
*3. Import a device private key 
*4. Call to TLK_ODRM_VerifyDeviceRoBeforeInstall()  API with invalid RO XML structure (RO integrity)
*5. Verify that correct error was returned
*******************************************************************************/   
DxError_t TST_ATP_SEC_RO_PreProcessingWithInvalidRO(void)
{

    DxUint32_t TST_Error = 0;
    DxUint8_t index = 0; 
    TLK_ODRM_KMNGKeyRing_t keyRingDeviceStruct = {0};

    TLK_ODRM_KMNGKeyRing_t keyRingKmacKrecStruct = {0};

    TLK_ODRM_KMNGKey_t  TST_KeyDevice = {0};
    TLK_ODRM_KMNGKey_t  TST_kmacKey = {0};
    TLK_ODRM_KMNGKey_t  TST_krecKey = {0};
    TLK_ODRM_Buffer_t   TST_RightsXML = {0};
    DxUint32_t  TST_KeyRingSize = 0;

    KMNG_UserSpecificKeyData_t  TST_UserSpecificKeyData = {0,0};
    TLK_SCLK_ServiceClockDescriptor_t    TST_ServiceDescr = {0};

    DxUint32_t TST_NumOfSymKeys = 0;
    DxUint32_t TST_NumOfRSAKeys = 0;
    DxUint32_t TST_NumOfDHKeys  = 0;

    /*******************Set password to RI ID****************/
    TLK_ODRM_Buffer_t passwordBuff;
    passwordBuff.buff_ptr   = TST_RiIDKey;
    passwordBuff.buffSizeInBytes = DX_VOS_StrLen(TST_RiIDKey);

    /*Initialize invalid RO XML structure*/
    TST_RightsXML.buffSizeInBytes = DX_VOS_StrLen(InvIntegrityRightsXmlStr);
    TST_RightsXML.buff_ptr = (DxByte*)InvIntegrityRightsXmlStr;

    TST_KRING_Clean();


    ATP_LOG_TST_PRINT((MW_ATP_MSG,"\n***************************************************** \n"));
    ATP_LOG_TST_PRINT((MW_ATP_MSG,"***** TST_ATP_SEC_RO_PreProcessingWithInvalidRO *****\n"));     
    ATP_LOG_TST_PRINT((MW_ATP_MSG,"*****************************************************\n \n"));

    /* 1. Get work space size */
    TST_Error = TLK_ODRM_WorkspaceSizeGet(TLK_ODRM_MIN_WORKSPACE, &RO_TST_gWorkspaceSizeInBytes);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_ODRM_WorkspaceSizeGet",
                               "TST_ATP_SEC_RO_PreProcessingWithInvalidRO", 
                               "TST_ATP_SEC_RO_PreProcessingWithInvalidRO");


    if(RO_TST_gWorkspaceSizeInBytes < ODRMTLK_TST_WORKSPACE_SIZE)
        TST_Error = TST_FAIL;

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "Insufficient Workspace Size",
                               "TST_ATP_SEC_RO_PreProcessingWithInvalidRO",
                               "TST_ATP_SEC_RO_PreProcessingWithInvalidRO");


    /* 2. Create key ring and initialize it with device private key */
    TST_NumOfSymKeys = 0;
    TST_NumOfRSAKeys = 1;
    TST_NumOfDHKeys  = 0;
    TST_Error =  KMNG_KeyRingInit(passwordBuff.buffSizeInBytes,
                                  passwordBuff.buff_ptr,
                                  TST_NumOfSymKeys,
                                  TST_NumOfRSAKeys,
                                  TST_NumOfDHKeys,
                                  TST_gKeyRingDevice);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "KMNG_KeyRingInit - TST_gKeyRingDevice",
                               "TST_ATP_SEC_RO_PreProcessingWithInvalidRO",
                               "TST_ATP_SEC_RO_PreProcessingWithInvalidRO");

    TST_NumOfSymKeys = 2;
    TST_NumOfRSAKeys = 0;
    TST_NumOfDHKeys  = 0;
    TST_Error = KMNG_KeyRingInit(passwordBuff.buffSizeInBytes,
                                 passwordBuff.buff_ptr,
                                 TST_NumOfSymKeys,
                                 TST_NumOfRSAKeys,
                                 TST_NumOfDHKeys,
                                 TST_gKeyRingKmacKrec);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "KMNG_KeyRingInit - TST_gKeyRingKmacKrec",
                               "TST_ATP_SEC_RO_PreProcessingWithInvalidRO",
                               "TST_ATP_SEC_RO_PreProcessingWithInvalidRO");

    TST_KeyDevice.keyPassword_ptr =passwordBuff.buff_ptr;
    TST_KeyDevice.keyPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    TST_kmacKey.keyPassword_ptr =passwordBuff.buff_ptr;
    TST_kmacKey.keyPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    TST_krecKey.keyPassword_ptr =passwordBuff.buff_ptr;
    TST_krecKey.keyPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    /*3. Import a device private key*/
    TST_Error = KMNG_ImportRsaUserKey(passwordBuff.buff_ptr, 
                                      passwordBuff.buffSizeInBytes,
                                      passwordBuff.buff_ptr, 
                                      passwordBuff.buffSizeInBytes,
                                      KMNG_KeyTypeRSAPair,                    /*keyType*/
                                      ODRMTLK_TST_RSA_1024_KEY_SIZE_IN_BYTES, /*keySize*/ 
                                      KMNG_KEY_USAGE_ODRM | KMNG_KEY_USAGE_SIGNING,                    /*keyUsage*/
                                      TLK_ODRM_KEY_RESTRICTION,               /*keyRestriction*/
                                      TST_UserSpecificKeyData,                    /*UserSpecificKeyData*/
                                      PrivModulus,                           /*modulus_ptr*/
                                      MODULUS_SIZE,                       /*modulusSize*/
                                      PubExponent,                       /*E_ptr*/
                                      PUB_EXP_SIZE,                    /*E_Size*/
                                      PrivExponent,                       /*D_ptr*/
                                      PRIV_EXP_SIZE,                   /*D_Size*/
                                      &TST_KeyDevice.keyId,
                                      TST_gKeyRingDevice);


    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "KMNG_ImportRsaUserKey ",
                               "TST_ATP_SEC_RO_PreProcessingWithInvalidRO",
                               "TST_ATP_SEC_RO_PreProcessingWithInvalidRO");

    keyRingDeviceStruct.keyKmngRing_ptr = TST_gKeyRingDevice;
    keyRingDeviceStruct.keyKmngRingPassword_ptr = passwordBuff.buff_ptr;
    keyRingDeviceStruct.keyKmngRingPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    keyRingKmacKrecStruct.keyKmngRing_ptr = TST_gKeyRingKmacKrec;
    keyRingKmacKrecStruct.keyKmngRingPassword_ptr = passwordBuff.buff_ptr;
    keyRingKmacKrecStruct.keyKmngRingPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    /*4. Call to  TLK_ODRM_VerifyDeviceRoBeforeInstall() API with invalid RO structure */
    TST_Error = TLK_ODRM_VerifyDeviceRoBeforeInstall(&TST_RightsXML,
                                                     &keyRingDeviceStruct,
                                                     &TST_KeyDevice,
                                                     &keyRingKmacKrecStruct,
                                                     &TST_kmacKey,
                                                     &TST_krecKey,
                                                     RO_TST_gWorkspace,
                                                     RO_TST_gWorkspaceSizeInBytes);
    /*5. Verify that TLK_ODRM_RC_ERROR_RO_MAC_VERIFICATION_FAILED error was returned */
    if(TST_Error != TLK_ODRM_RC_ERROR_RO_MAC_VERIFICATION_FAILED)
        TST_Error = TST_FAIL;
    else
        TST_Error = DX_SUCCESS;
    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_ODRM_VerifyDeviceRoBeforeInstall with invalid RO. Test ",
                               "TST_ATP_SEC_RO_PreProcessingWithInvalidRO",
                               "TST_ATP_SEC_RO_PreProcessingWithInvalidRO");

EXIT_ON_ERROR:
    TLK_SCLK_Terminate(RO_TST_gWorkspace, RO_TST_gWorkspaceSizeInBytes);
    return TST_Error;
}

/*****************************************************************************
* Function Name:                                                           
*  TST_ATP_SEC_RO_DomainRoInstWithInvalidFlag
* 
* Inputs:
*  None
*
* Outputs:
*  DxError_t - Function error return                                        
*
* TEST DESCRIPTION:
*  To test license pre-processing with domain key  using invalid usage flag
* Algorithm:
* 1. Get workspace size. 
* 2. Create key ring and initialize it.
* 3. Import domain key using invalid usage flag
* 4. Call to TLK_ODRM_VerifyDomainRoBeforeInstall() API with Domain key from step 3.
* 5. Verify that TLK_ODRM_RC_ERROR_INCORRECT_KEY_PERMISSIONS error was returned
*******************************************************************************/   
DxError_t TST_ATP_SEC_RO_DomainRoInstWithInvalidFlag(void)
{
    DxUint32_t TST_Error = 0;
    DxUint8_t index = 0; 
    TLK_ODRM_KMNGKeyRing_t keyRingDeviceStruct = {0};
    TLK_ODRM_KMNGKeyRing_t keyRingDomainStruct = {0};
    TLK_ODRM_KMNGKeyRing_t keyRingKmacKrecStruct = {0};

    TLK_ODRM_KMNGKey_t  TST_KeyDomain = {0};
    TLK_ODRM_KMNGKey_t  TST_kmacKey = {0};
    TLK_ODRM_KMNGKey_t  TST_krecKey = {0};
    TLK_ODRM_Buffer_t   TST_RightsXML = {0};
    DxUint32_t  TST_KeyRingSize = 0;

    KMNG_UserSpecificKeyData_t  TST_UserSpecificKeyData = {0,0};
    TLK_SCLK_ServiceClockDescriptor_t    TST_ServiceDescr = {0};

    DxUint32_t TST_NumOfSymKeys = 0;
    DxUint32_t TST_NumOfRSAKeys = 0;
    DxUint32_t TST_NumOfDHKeys  = 0;

    /*******************Set password to RI ID****************/
    TLK_ODRM_Buffer_t passwordBuff;
    passwordBuff.buff_ptr        = TST_RiIDKey;
    passwordBuff.buffSizeInBytes = DX_VOS_StrLen(TST_RiIDKey);
    TST_KRING_Clean();

    ATP_LOG_TST_PRINT((MW_ATP_MSG,"\n****************************************************** \n"));
    ATP_LOG_TST_PRINT((MW_ATP_MSG,"***** TST_ATP_SEC_RO_DomainRoInstWithInvalidFlag *****\n"));     
    ATP_LOG_TST_PRINT((MW_ATP_MSG,"******************************************************\n \n"));


    /*Initialize rights  struct*/
    TST_RightsXML.buffSizeInBytes = DX_VOS_StrLen(DomainRoXml);
    TST_RightsXML.buff_ptr = (DxByte*)DomainRoXml;

    /* 1. Get work space size */
    TST_Error = TLK_ODRM_WorkspaceSizeGet(TLK_ODRM_MIN_WORKSPACE, &RO_TST_gWorkspaceSizeInBytes);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_ODRM_WorkspaceSizeGet",
                               "TST_ATP_SEC_RO_DomainRoInstWithInvalidFlag", 
                               "TST_ATP_SEC_RO_DomainRoInstWithInvalidFlag");


    if(RO_TST_gWorkspaceSizeInBytes < ODRMTLK_TST_WORKSPACE_SIZE)
        TST_Error = TST_FAIL;

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "Insufficient Workspace Size",
                               "TST_ATP_SEC_RO_DomainRoInstWithInvalidFlag",
                               "TST_ATP_SEC_RO_DomainRoInstWithInvalidFlag");

    /* 2. Create key ring and initialize it.*/
    TST_NumOfSymKeys = 2;
    TST_NumOfRSAKeys = 0;
    TST_NumOfDHKeys  = 0;
    TST_Error = KMNG_KeyRingInit(passwordBuff.buffSizeInBytes,
                                 passwordBuff.buff_ptr,
                                 TST_NumOfSymKeys,
                                 TST_NumOfRSAKeys,
                                 TST_NumOfDHKeys,
                                 TST_gKeyRingKmacKrec);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "KMNG_KeyRingInit - TST_gKeyRingKmacKrec",
                               "TST_ATP_SEC_RO_DomainRoInstWithInvalidFlag",
                               "TST_ATP_SEC_RO_DomainRoInstWithInvalidFlag");

    TST_NumOfSymKeys = 1;
    TST_NumOfRSAKeys = 0;
    TST_NumOfDHKeys  = 0;
    TST_Error =  KMNG_KeyRingInit(passwordBuff.buffSizeInBytes,
                                  passwordBuff.buff_ptr,
                                  TST_NumOfSymKeys,
                                  TST_NumOfRSAKeys,
                                  TST_NumOfDHKeys,
                                  TST_gKeyRingDomain);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "KMNG_KeyRingInit - TST_gKeyRingDomain",
                               "TST_ATP_SEC_RO_DomainRoInstWithInvalidFlag",
                               "TST_ATP_SEC_RO_DomainRoInstWithInvalidFlag");

    TST_KeyDomain.keyPassword_ptr =passwordBuff.buff_ptr;
    TST_KeyDomain.keyPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    TST_kmacKey.keyPassword_ptr =passwordBuff.buff_ptr;
    TST_kmacKey.keyPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    TST_krecKey.keyPassword_ptr =passwordBuff.buff_ptr;
    TST_krecKey.keyPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    /* 3. Import domain key using invalid usage flag*/
    TST_Error = KMNG_ImportSymUserKey(passwordBuff.buff_ptr, 
                                      passwordBuff.buffSizeInBytes,
                                      passwordBuff.buff_ptr, 
                                      passwordBuff.buffSizeInBytes,
                                      KMNG_KeyTypeAES,
                                      ODRMTLK_TST_AES_KEY_SIZE,
                                      KMNG_KEY_USAGE_LEGACY,//invalid flag
                                      TLK_ODRM_KEY_RESTRICTION,
                                      TST_UserSpecificKeyData, 
                                      SecretDomainKey,
                                      &TST_KeyDomain.keyId,
                                      TST_gKeyRingDomain);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "KMNG_ImportSymUserKey",
                               "TST_ATP_SEC_RO_DomainRoInstWithInvalidFlag",
                               "TST_ATP_SEC_RO_DomainRoInstWithInvalidFlag");

    keyRingDomainStruct.keyKmngRing_ptr = TST_gKeyRingDomain;
    keyRingDomainStruct.keyKmngRingPassword_ptr = passwordBuff.buff_ptr;
    keyRingDomainStruct.keyKmngRingPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    keyRingKmacKrecStruct.keyKmngRing_ptr = TST_gKeyRingKmacKrec;
    keyRingKmacKrecStruct.keyKmngRingPassword_ptr = passwordBuff.buff_ptr;
    keyRingKmacKrecStruct.keyKmngRingPasswordLenInBytes = passwordBuff.buffSizeInBytes;


    /*4. Call to  TLK_ODRM_VerifyDomainRoBeforeInstall() API with domain key . */
    TST_Error = TLK_ODRM_VerifyDomainRoBeforeInstall(&TST_RightsXML,
                                                     &keyRingDomainStruct,
                                                     &TST_KeyDomain,
                                                     0,
                                                     &keyRingKmacKrecStruct,
                                                     &TST_kmacKey,
                                                     &TST_krecKey,
                                                     RO_TST_gWorkspace,
                                                     RO_TST_gWorkspaceSizeInBytes);
    /*5. Verify that TLK_ODRM_RC_ERROR_INCORRECT_KEY_PERMISSIONS error was returned */
    if(TST_Error != TLK_ODRM_RC_ERROR_INCORRECT_KEY_PERMISSIONS)
        TST_Error = TST_FAIL;
    else
        TST_Error = DX_SUCCESS;

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_ODRM_VerifyDomainRoBeforeInstall succeed with invalid usage flag. Test ",
                               "TST_ATP_SEC_RO_DomainRoInstWithInvalidFlag",
                               "TST_ATP_SEC_RO_DomainRoInstWithInvalidFlag");

EXIT_ON_ERROR:
    return TST_Error;
}

/*****************************************************************************
* Function Name:                                                           
*  TST_ATP_SEC_RO_DomainRoInstWithInvalidKey
* 
* Inputs:
*  None
*
* Outputs:
*  DxError_t - Function error return                                        
*
* TEST DESCRIPTION:
* To test license pre-processing with invalid domain key 
* Algorithm:
* 1. Get workspace size. 
* 2. Create key ring and initialize it.
* 3. Import invalid domain key 
* 4. Call to TLK_ODRM_VerifyDomainRoBeforeInstall() API with Domain key from step 3.
* 5. Verify that TLK_ODRM_RC_ERROR_CRYS_KMNG error was returned
*******************************************************************************/   
DxError_t TST_ATP_SEC_RO_DomainRoInstWithInvalidKey(void)
{
    DxUint32_t TST_Error = 0;
    DxUint8_t index = 0; 
    TLK_ODRM_KMNGKeyRing_t keyRingDeviceStruct = {0};
    TLK_ODRM_KMNGKeyRing_t keyRingDomainStruct = {0};
    TLK_ODRM_KMNGKeyRing_t keyRingKmacKrecStruct = {0};

    TLK_ODRM_KMNGKey_t  TST_KeyDomain = {0};
    TLK_ODRM_KMNGKey_t  TST_kmacKey = {0};
    TLK_ODRM_KMNGKey_t  TST_krecKey = {0};
    TLK_ODRM_Buffer_t   TST_RightsXML = {0};
    DxUint32_t  TST_KeyRingSize = 0;

    KMNG_UserSpecificKeyData_t  TST_UserSpecificKeyData = {0,0};
    TLK_SCLK_ServiceClockDescriptor_t    TST_ServiceDescr = {0};

    DxUint32_t TST_NumOfSymKeys = 0;
    DxUint32_t TST_NumOfRSAKeys = 0;
    DxUint32_t TST_NumOfDHKeys  = 0;

    /*******************Set password to RI ID ****************/
    TLK_ODRM_Buffer_t passwordBuff;
    passwordBuff.buff_ptr        = TST_RiIDKey;
    passwordBuff.buffSizeInBytes = DX_VOS_StrLen(TST_RiIDKey);
    TST_KRING_Clean();

    ATP_LOG_TST_PRINT((MW_ATP_MSG,"\n************************************************* \n"));
    ATP_LOG_TST_PRINT((MW_ATP_MSG,"****TST_ATP_SEC_RO_DomainRoInstWithInvalidKey****\n"));     
    ATP_LOG_TST_PRINT((MW_ATP_MSG,"*************************************************\n \n"));


    /*Initialize rights  struct*/
    TST_RightsXML.buffSizeInBytes = DX_VOS_StrLen(DomainRoXml);
    TST_RightsXML.buff_ptr = (DxByte*)DomainRoXml;

    /* 1. Get work space size */
    TST_Error = TLK_ODRM_WorkspaceSizeGet(TLK_ODRM_MIN_WORKSPACE, &RO_TST_gWorkspaceSizeInBytes);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_ODRM_WorkspaceSizeGet",
                               "TST_ATP_SEC_RO_DomainRoInstWithInvalidKey", 
                               "TST_ATP_SEC_RO_DomainRoInstWithInvalidKey");


    if(RO_TST_gWorkspaceSizeInBytes < ODRMTLK_TST_WORKSPACE_SIZE)
        TST_Error = TST_FAIL;

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "Insufficient Workspace Size",
                               "TST_ATP_SEC_RO_DomainRoInstWithInvalidKey",
                               "TST_ATP_SEC_RO_DomainRoInstWithInvalidKey");

    /* 2. Create key rings and initialize it.*/
    TST_NumOfSymKeys = 2;
    TST_NumOfRSAKeys = 0;
    TST_NumOfDHKeys  = 0;
    TST_Error = KMNG_KeyRingInit(passwordBuff.buffSizeInBytes,
                                 passwordBuff.buff_ptr,
                                 TST_NumOfSymKeys,
                                 TST_NumOfRSAKeys,
                                 TST_NumOfDHKeys,
                                 TST_gKeyRingKmacKrec);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "KMNG_KeyRingInit - TST_gKeyRingKmacKrec",
                               "TST_ATP_SEC_RO_DomainRoInstWithInvalidKey",
                               "TST_ATP_SEC_RO_DomainRoInstWithInvalidKey");

    TST_NumOfSymKeys = 1;
    TST_NumOfRSAKeys = 0;
    TST_NumOfDHKeys  = 0;
    TST_Error =  KMNG_KeyRingInit(passwordBuff.buffSizeInBytes,
                                  passwordBuff.buff_ptr,
                                  TST_NumOfSymKeys,
                                  TST_NumOfRSAKeys,
                                  TST_NumOfDHKeys,
                                  TST_gKeyRingDomain);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "KMNG_KeyRingInit - TST_gKeyRingDomain",
                               "TST_ATP_SEC_RO_DomainRoInstWithInvalidKey",
                               "TST_ATP_SEC_RO_DomainRoInstWithInvalidKey");

    TST_KeyDomain.keyPassword_ptr =passwordBuff.buff_ptr;
    TST_KeyDomain.keyPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    TST_kmacKey.keyPassword_ptr =passwordBuff.buff_ptr;
    TST_kmacKey.keyPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    TST_krecKey.keyPassword_ptr =passwordBuff.buff_ptr;
    TST_krecKey.keyPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    /* 3. Import invalid domain key */
    TST_Error = KMNG_ImportSymUserKey(passwordBuff.buff_ptr, 
                                      passwordBuff.buffSizeInBytes,
                                      passwordBuff.buff_ptr, 
                                      passwordBuff.buffSizeInBytes,
                                      KMNG_KeyTypeAES,
                                      ODRMTLK_TST_AES_KEY_SIZE,
                                      KMNG_KEY_USAGE_ODRM ,
                                      TLK_ODRM_KEY_RESTRICTION,
                                      TST_UserSpecificKeyData, 
                                      InvSecretDomainKey,//invalid key
                                      &TST_KeyDomain.keyId,
                                      TST_gKeyRingDomain);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "KMNG_ImportSymUserKey",
                               "TST_ATP_SEC_RO_DomainRoInstWithInvalidKey",
                               "TST_ATP_SEC_RO_DomainRoInstWithInvalidKey");

    keyRingDomainStruct.keyKmngRing_ptr = TST_gKeyRingDomain;
    keyRingDomainStruct.keyKmngRingPassword_ptr = passwordBuff.buff_ptr;
    keyRingDomainStruct.keyKmngRingPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    keyRingKmacKrecStruct.keyKmngRing_ptr = TST_gKeyRingKmacKrec;
    keyRingKmacKrecStruct.keyKmngRingPassword_ptr = passwordBuff.buff_ptr;
    keyRingKmacKrecStruct.keyKmngRingPasswordLenInBytes = passwordBuff.buffSizeInBytes;

  
    /*4. Call to  TLK_ODRM_VerifyDeviceRoBeforeInstall() API with the invalid domain secret key . */
    TST_Error = TLK_ODRM_VerifyDomainRoBeforeInstall(&TST_RightsXML,
                                                     &keyRingDomainStruct,
                                                     &TST_KeyDomain,
                                                     0,
                                                     &keyRingKmacKrecStruct,
                                                     &TST_kmacKey,
                                                     &TST_krecKey,
                                                     RO_TST_gWorkspace,
                                                     RO_TST_gWorkspaceSizeInBytes);
    /*5. Verify that TLK_ODRM_RC_ERROR_CRYS_KMNG error was returned */
    if(TST_Error != TLK_ODRM_RC_ERROR_CRYS_KMNG)
        TST_Error = TST_FAIL;
    else
        TST_Error = DX_SUCCESS;

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_ODRM_VerifyDomainRoBeforeInstall  with invalid domain key. Test ",
                               "TST_ATP_SEC_RO_DomainRoInstWithInvalidKey",
                               "TST_ATP_SEC_RO_DomainRoInstWithInvalidKey");

EXIT_ON_ERROR:
    return TST_Error;
}

/*****************************************************************************
* Function Name:                                                           
*  TST_ATP_SEC_RO_DomainRoInstWithInvPassword
* 
* Inputs:
*  None
*
* Outputs:
*  DxError_t - Function error return                                        
*
* TEST DESCRIPTION:
*  To test license pre-processing with domain key using invalid key ring password 
* Algorithm:
* 1.Get workspace size. 
* 2.Create key ring and initialize it with not RI ID password.
* 3.Import domain key 
* 4.Call to TLK_ODRM_VerifyDomainRoBeforeInstall() API with Domain key from step 3.
* 5.Verify that TLK_ODRM_RC_ERROR_KMNG error was returned
*******************************************************************************/   
DxError_t TST_ATP_SEC_RO_DomainRoInstWithInvPassword(void)
{
    DxUint32_t TST_Error = 0;
    DxUint8_t index = 0; 
    TLK_ODRM_KMNGKeyRing_t keyRingDeviceStruct = {0};
    TLK_ODRM_KMNGKeyRing_t keyRingDomainStruct = {0};
    TLK_ODRM_KMNGKeyRing_t keyRingKmacKrecStruct = {0};

    TLK_ODRM_KMNGKey_t  TST_KeyDomain = {0};
    TLK_ODRM_KMNGKey_t  TST_kmacKey = {0};
    TLK_ODRM_KMNGKey_t  TST_krecKey = {0};
    TLK_ODRM_Buffer_t   TST_RightsXML = {0};
    DxUint32_t  TST_KeyRingSize = 0;

    KMNG_UserSpecificKeyData_t  TST_UserSpecificKeyData = {0,0};
    TLK_SCLK_ServiceClockDescriptor_t    TST_ServiceDescr = {0};

    DxUint32_t TST_NumOfSymKeys = 0;
    DxUint32_t TST_NumOfRSAKeys = 0;
    DxUint32_t TST_NumOfDHKeys  = 0;

    /*******************Set password to RI ID****************/
    TLK_ODRM_Buffer_t passwordBuff;
    TLK_ODRM_Buffer_t InvPasswordBuff;
    passwordBuff.buff_ptr        = TST_RiIDKey;
    passwordBuff.buffSizeInBytes = DX_VOS_StrLen(TST_RiIDKey);

    InvPasswordBuff.buff_ptr        = TST_InvRiIDKey;
    InvPasswordBuff.buffSizeInBytes = DX_VOS_StrLen(TST_InvRiIDKey);

    TST_KRING_Clean();

    ATP_LOG_TST_PRINT((MW_ATP_MSG,"\n************************************************** \n"));
    ATP_LOG_TST_PRINT((MW_ATP_MSG,"*** TST_ATP_SEC_RO_DomainRoInstWithInvPassword ***\n"));     
    ATP_LOG_TST_PRINT((MW_ATP_MSG,"**************************************************\n \n"));


    /*Initialize rights  struct*/
    TST_RightsXML.buffSizeInBytes = DX_VOS_StrLen(DomainRoXml);
    TST_RightsXML.buff_ptr = (DxByte*)DomainRoXml;

    /* 1. Get work space size */
    TST_Error = TLK_ODRM_WorkspaceSizeGet(TLK_ODRM_MIN_WORKSPACE, &RO_TST_gWorkspaceSizeInBytes);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_ODRM_WorkspaceSizeGet",
                               "TST_ATP_SEC_RO_DomainRoInstWithInvPassword", 
                               "TST_ATP_SEC_RO_DomainRoInstWithInvPassword");


    if(RO_TST_gWorkspaceSizeInBytes < ODRMTLK_TST_WORKSPACE_SIZE)
        TST_Error = TST_FAIL;

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "Insufficient Workspace Size",
                               "TST_ATP_SEC_RO_DomainRoInstWithInvPassword",
                               "TST_ATP_SEC_RO_DomainRoInstWithInvPassword");


    TST_NumOfSymKeys = 2;
    TST_NumOfRSAKeys = 0;
    TST_NumOfDHKeys  = 0;

    TST_Error = KMNG_KeyRingInit(passwordBuff.buffSizeInBytes,
                                 passwordBuff.buff_ptr,
                                 TST_NumOfSymKeys,
                                 TST_NumOfRSAKeys,
                                 TST_NumOfDHKeys,
                                 TST_gKeyRingKmacKrec);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "KMNG_KeyRingInit - TST_gKeyRingKmacKrec",
                               "TST_ATP_SEC_RO_DomainRoInstWithInvPassword",
                               "TST_ATP_SEC_RO_DomainRoInstWithInvPassword");

    /* 2.Create key ring and initialize it with not RI ID password.*/
    TST_NumOfSymKeys = 1;
    TST_NumOfRSAKeys = 0;
    TST_NumOfDHKeys  = 0;

    TST_Error =  KMNG_KeyRingInit(InvPasswordBuff.buffSizeInBytes,
                                  InvPasswordBuff.buff_ptr,
                                  TST_NumOfSymKeys,
                                  TST_NumOfRSAKeys,
                                  TST_NumOfDHKeys,
                                  TST_gKeyRingDomain);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "KMNG_KeyRingInit - TST_gKeyRingDomain",
                               "TST_ATP_SEC_RO_DomainRoInstWithInvPassword",
                               "TST_ATP_SEC_RO_DomainRoInstWithInvPassword");

    TST_KeyDomain.keyPassword_ptr =InvPasswordBuff.buff_ptr;
    TST_KeyDomain.keyPasswordLenInBytes = InvPasswordBuff.buffSizeInBytes;

    TST_kmacKey.keyPassword_ptr =passwordBuff.buff_ptr;
    TST_kmacKey.keyPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    TST_krecKey.keyPassword_ptr =passwordBuff.buff_ptr;
    TST_krecKey.keyPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    /* 3.Import domain key*/
    TST_Error = KMNG_ImportSymUserKey(InvPasswordBuff.buff_ptr, //invalid password
                                      InvPasswordBuff.buffSizeInBytes,
                                      passwordBuff.buff_ptr, 
                                      passwordBuff.buffSizeInBytes,
                                      KMNG_KeyTypeAES,
                                      ODRMTLK_TST_AES_KEY_SIZE,
                                      KMNG_KEY_USAGE_ODRM,
                                      TLK_ODRM_KEY_RESTRICTION,
                                      TST_UserSpecificKeyData, 
                                      SecretDomainKey,
                                      &TST_KeyDomain.keyId,
                                      TST_gKeyRingDomain);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "KMNG_ImportSymUserKey",
                               "TST_ATP_SEC_RO_DomainRoInstWithInvPassword",
                               "TST_ATP_SEC_RO_DomainRoInstWithInvPassword");

    keyRingDomainStruct.keyKmngRing_ptr = TST_gKeyRingDomain;
    keyRingDomainStruct.keyKmngRingPassword_ptr = passwordBuff.buff_ptr;
    keyRingDomainStruct.keyKmngRingPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    keyRingKmacKrecStruct.keyKmngRing_ptr = TST_gKeyRingKmacKrec;
    keyRingKmacKrecStruct.keyKmngRingPassword_ptr = passwordBuff.buff_ptr;
    keyRingKmacKrecStruct.keyKmngRingPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    /*4. Call to  VerifyDomainRoBeforeInstall() API with the domain key from step 3. */
    TST_Error = TLK_ODRM_VerifyDomainRoBeforeInstall(&TST_RightsXML,
                                                     &keyRingDomainStruct,
                                                     &TST_KeyDomain,
                                                     0,
                                                     &keyRingKmacKrecStruct,
                                                     &TST_kmacKey,
                                                     &TST_krecKey,
                                                     RO_TST_gWorkspace,
                                                     RO_TST_gWorkspaceSizeInBytes);
    /*6. Verify that TLK_ODRM_RC_ERROR_KMNG error was returned */
    if(TST_Error != TLK_ODRM_RC_ERROR_KMNG)
        TST_Error = TST_FAIL;
    else
        TST_Error = DX_SUCCESS;

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_ODRM_VerifyDomainRoBeforeInstall with invalid password. Test ",
                               "TST_ATP_SEC_RO_DomainRoInstWithInvPassword",
                               "TST_ATP_SEC_RO_DomainRoInstWithInvPassword");

EXIT_ON_ERROR:
    return TST_Error;
}
